<?php

namespace App\Models;

use App\Core\Database;
use App\Models\Entities\RezervaceEntity;

/**
 * Reservation Model – správa rezervací ledu
 * Pracuje s hodinovými sloty: hour_start až hour_end
 */
class Reservation
{
    private Database $db;

    public function __construct(Database $db)
    {
        $this->db = $db;
    }

    /**
     * Vrátí všechny rezervace na daný den jako entity objekty
     * @return RezervaceEntity[]
     */
    public function getByDate($date)
    {
        $stmt = $this->db->pdo()->prepare(
            "SELECT * FROM rezervace WHERE date = ? ORDER BY hour_start"
        );
        $stmt->execute([$date]);
        $rows = $stmt->fetchAll(\PDO::FETCH_ASSOC);
        
        return array_map(function($row) {
            $entity = new RezervaceEntity();
            $entity->fill($row);
            return $entity;
        }, $rows);
    }

    /**
     * Vrátí konkrétní rezervaci jako entity objekt
     */
    public function getById($id): ?RezervaceEntity
    {
        $stmt = $this->db->pdo()->prepare(
            "SELECT * FROM rezervace WHERE id = ?"
        );
        $stmt->execute([$id]);
        $row = $stmt->fetch(\PDO::FETCH_ASSOC);
        
        if (!$row) {
            return null;
        }
        
        $entity = new RezervaceEntity();
        $entity->fill($row);
        return $entity;
    }

    /**
     * Vytvoří novou rezervaci
     * Vrací true pokud OK, jinak false
     */
    public function create($date, $hour_start, $hour_end, $title)
    {
        // Validace: hour_end musí být větší než hour_start
        if ($hour_end <= $hour_start) {
            return false;
        }

        // Ověřit, že hodin nejsou obsazené
        if ($this->hasConflict($date, $hour_start, $hour_end)) {
            return false;
        }

        $stmt = $this->db->pdo()->prepare(
            "INSERT INTO rezervace (date, hour_start, hour_end, title)
             VALUES (?, ?, ?, ?)"
        );
        return $stmt->execute([$date, $hour_start, $hour_end, $title]);
    }

    /**
     * Smaže rezervaci
     */
    public function delete($id)
    {
        $stmt = $this->db->pdo()->prepare(
            "DELETE FROM rezervace WHERE id = ?"
        );
        return $stmt->execute([$id]);
    }

    /**
     * Ověří, zda se daný čas překrývá s existující rezervací
     * Vrací true pokud je konflikt
     */
    public function hasConflict($date, $hour_start, $hour_end, $excludeId = null)
    {
        // Kontrolujeme: existuje-li rezervace, jejichž časy se překrývají
        // Dva časové intervaly se překrývají, pokud:
        // start1 < end2 AND start2 < end1

        $query = "SELECT COUNT(*) as cnt FROM rezervace 
                  WHERE date = ? 
                    AND hour_start < ? 
                    AND hour_end > ?";
        $params = [$date, $hour_end, $hour_start];

        if ($excludeId) {
            $query .= " AND id != ?";
            $params[] = $excludeId;
        }

        $stmt = $this->db->pdo()->prepare($query);
        $stmt->execute($params);
        $result = $stmt->fetch(\PDO::FETCH_ASSOC);

        return $result['cnt'] > 0;
    }

    /**
     * Vrátí pole zarezervovaných hodin pro daný den jako entity
     * Používáno pro UI – která politička jsou zarezervovaná
     * Vrací array: [hour => RezervaceEntity]
     * @return RezervaceEntity[]
     */
    public function getOccupiedHours($date)
    {
        $reservations = $this->getByDate($date);
        $occupied = [];

        foreach ($reservations as $res) {
            for ($h = $res->hour_start; $h < $res->hour_end; $h++) {
                $occupied[$h] = $res;
            }
        }

        return $occupied;
    }

    /**
     * Aktualizuje rezervaci
     */
    public function update($id, $date, $hour_start, $hour_end, $title)
    {
        if ($hour_end <= $hour_start) {
            return false;
        }

        // Ověřit konflikt, ale vylučujeme tuto rezervaci
        if ($this->hasConflict($date, $hour_start, $hour_end, $id)) {
            return false;
        }

        $stmt = $this->db->pdo()->prepare(
            "UPDATE rezervace SET date = ?, hour_start = ?, hour_end = ?, title = ? WHERE id = ?"
        );
        return $stmt->execute([$date, $hour_start, $hour_end, $title, $id]);
    }
}

